
--
-- Definition des proprietes de DB
--
CREATE TABLE cec_db_properties
(
 dbp_name VARCHAR(255) NOT NULL,
 dbp_value VARCHAR(255) NOT NULL,
 dbp_description VARCHAR(1023)
);
ALTER TABLE cec_db_properties ADD PRIMARY KEY (dbp_name);


--
-- Definition des licenses
--
CREATE TABLE cec_licenses(
 lic_id INTEGER NOT NULL,
 lic_company VARCHAR(50) NOT NULL,
 lic_module VARCHAR(50) NOT NULL,
 lic_version VARCHAR(50) NOT NULL,
 lic_host VARCHAR(1023),
 lic_token VARCHAR(50),
 lic_expire VARCHAR(50)
);
ALTER TABLE cec_licenses ADD PRIMARY KEY (lic_id);
ALTER TABLE cec_licenses ADD CONSTRAINT cec_unq_lic_id UNIQUE (lic_id);



--
-- Definition des groups
--
CREATE TABLE  cec_groups(
  grp_id INTEGER NOT NULL,
  grp_name VARCHAR(255) NOT NULL,
  grp_description VARCHAR(1023) DEFAULT NULL,
  grp_flags INTEGER DEFAULT 0 NOT NULL
);
ALTER TABLE cec_groups ADD PRIMARY KEY (grp_id);
ALTER TABLE cec_groups ADD CONSTRAINT cec_unq_grp_id UNIQUE (grp_id);
ALTER TABLE cec_groups ADD CONSTRAINT cec_unq_grp_name UNIQUE (grp_name);


--
-- Definition des users
--
CREATE TABLE cec_users(
 usr_id INTEGER NOT NULL,
 usr_name VARCHAR(255) NOT NULL,
 usr_login VARCHAR(255) NOT NULL,
 usr_password VARCHAR(255) NOT NULL,
 usr_default_grp_id INTEGER,
 usr_description VARCHAR(1023),
 usr_flags INTEGER DEFAULT 0 NOT NULL
);
ALTER TABLE cec_users ADD PRIMARY KEY (usr_id);
ALTER TABLE cec_users ADD CONSTRAINT cec_unq_usr_id UNIQUE (usr_id);
ALTER TABLE cec_users ADD CONSTRAINT cec_unq_usr_name UNIQUE (usr_name);
ALTER TABLE cec_users ADD CONSTRAINT cec_unq_usr_login UNIQUE (usr_login);

ALTER TABLE cec_users ADD CONSTRAINT cec_fk_usr_def_grp FOREIGN KEY (usr_default_grp_id) REFERENCES cec_groups (grp_id) ON DELETE CASCADE;

--
-- Definition de la relation entre les groups et les users
--
CREATE TABLE cec_groups_users(
 grp_grp_id INTEGER NOT NULL,
 grp_usr_id INTEGER NOT NULL
);
ALTER TABLE cec_groups_users ADD PRIMARY KEY (grp_grp_id, grp_usr_id);
ALTER TABLE cec_groups_users ADD CONSTRAINT cec_unq_grp_usr UNIQUE (grp_grp_id, grp_usr_id);
CREATE INDEX cec_idx_grp_usr_id ON cec_groups_users(grp_usr_id);
CREATE INDEX cec_idx_grp_grp_id ON cec_groups_users(grp_grp_id);
-- ALTER TABLE cec_groups_users ADD KEY grp_usr_id;
-- ALTER TABLE cec_groups_users ADD KEY grp_grp_id;
ALTER TABLE cec_groups_users ADD CONSTRAINT cec_fk_grp_grp FOREIGN KEY (grp_grp_id) REFERENCES cec_groups (grp_id) ON DELETE CASCADE;
ALTER TABLE cec_groups_users ADD CONSTRAINT cec_fk_grp_usr FOREIGN KEY (grp_usr_id) REFERENCES cec_users (usr_id) ON DELETE CASCADE;


--
-- Definition des sessions
--
CREATE TABLE cec_sessions(
 ses_id INTEGER NOT NULL,
 ses_usr_id INTEGER NOT NULL,
 ses_module VARCHAR(10) NOT NULL,
 ses_flags INTEGER DEFAULT 0 NOT NULL,
 ses_expire INTEGER NOT NULL
);
ALTER TABLE cec_sessions ADD PRIMARY KEY (ses_id);
ALTER TABLE cec_sessions ADD CONSTRAINT cec_unq_ses_id UNIQUE (ses_id);
CREATE INDEX cec_idx_ses_usr_id ON cec_sessions(ses_usr_id);
ALTER TABLE cec_sessions ADD CONSTRAINT cec_fk_ses_usr FOREIGN KEY (ses_usr_id) REFERENCES cec_users(usr_id) ON DELETE CASCADE;


--
-- Definition des preferencies
--
CREATE TABLE cec_preferencies (
 upf_id INTEGER NOT NULL,
 upf_usr_id INTEGER NOT NULL,
 upf_key VARCHAR(255) NOT NULL,
 upf_flags INTEGER DEFAULT 0 NOT NULL,
 upf_data bytea NOT NULL,
 upf_data_size NUMERIC DEFAULT 0 NOT NULL
);
ALTER TABLE cec_preferencies ADD PRIMARY KEY (upf_id);
ALTER TABLE cec_preferencies ADD CONSTRAINT cec_unq_upf_id UNIQUE (upf_id);
ALTER TABLE cec_preferencies ADD CONSTRAINT cec_unq_preferencies UNIQUE (upf_usr_id, upf_key);
ALTER TABLE cec_preferencies ADD CONSTRAINT cec_fk_upf_usr FOREIGN KEY (upf_usr_id) REFERENCES cec_users(usr_id) ON DELETE CASCADE;
CREATE INDEX cec_idx_upf_usr_id ON cec_preferencies(upf_usr_id);


--
-- Definition des objects et de son etat (freeze, verrouillage)
--
CREATE TABLE cec_objects(
 obj_id INTEGER NOT NULL,
 obj_flags INTEGER DEFAULT 0 NOT NULL,
 obj_ses_id INTEGER DEFAULT NULL,
 obj_time_update TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
ALTER TABLE cec_objects ADD PRIMARY KEY (obj_id);
ALTER TABLE cec_objects ADD CONSTRAINT cec_unq_obj_id UNIQUE (obj_id);
ALTER TABLE cec_objects ADD CONSTRAINT cec_fk_obj_ses_id FOREIGN KEY (obj_ses_id) REFERENCES cec_sessions(ses_id);

--
-- Definition des modeles de donnees pour un object (regroupe les proprietes etendues)
--
CREATE TABLE cec_obj_models(
 mod_id INTEGER NOT NULL,
 mod_obj_id INTEGER,
 mod_nature VARCHAR(255) NOT NULL,
 mod_format VARCHAR(10),
 mod_encoding VARCHAR(20),
 --    ajout pour eviter de lire le flux afin de connaitre une compatibilite
 mod_crc VARCHAR(128), 
 mod_data bytea NOT NULL,
 mod_data_size NUMERIC DEFAULT 0 NOT NULL,
 --    sert a gerer une eventuelle annulation de la modifiation / pas reellement utilise actuellement
 mod_previous NUMERIC DEFAULT 0 NOT NULL
);
ALTER TABLE cec_obj_models ADD PRIMARY KEY (mod_id);
ALTER TABLE cec_obj_models ADD CONSTRAINT cec_unq_mod_id UNIQUE (mod_id);
ALTER TABLE cec_obj_models ADD CONSTRAINT cec_unq_models UNIQUE (mod_obj_id, mod_nature);
ALTER TABLE cec_obj_models ADD CONSTRAINT cec_fk_mod_obj_id FOREIGN KEY (mod_obj_id) REFERENCES cec_objects(obj_id) ON DELETE CASCADE;
CREATE INDEX cec_idx_mod_obj_id ON cec_obj_models(mod_obj_id);

--
-- Definition des proprietes simple pour un object
--
CREATE TABLE cec_obj_properties(
 opr_id INTEGER NOT NULL,
 opr_obj_id INTEGER NOT NULL,
 opr_name VARCHAR(255) NOT NULL,
 opr_value VARCHAR(255) NOT NULL,
 opr_type VARCHAR(50)
);
ALTER TABLE cec_obj_properties ADD PRIMARY KEY (opr_id);
ALTER TABLE cec_obj_properties ADD CONSTRAINT cec_unq_opr_id UNIQUE (opr_id);
ALTER TABLE cec_obj_properties ADD CONSTRAINT cec_unq_properties UNIQUE (opr_obj_id, opr_name);
ALTER TABLE cec_obj_properties ADD CONSTRAINT cec_fk_opr_obj_id FOREIGN KEY (opr_obj_id) REFERENCES cec_objects(obj_id) ON DELETE CASCADE;
CREATE INDEX cec_idx_opr_obj_id ON cec_obj_properties(opr_obj_id);


--
-- Definition des dossiers (folders)
--
CREATE TABLE cec_folders(
 fol_id INTEGER NOT NULL,
 fol_parent_id INTEGER,
 fol_obj_id INTEGER DEFAULT NULL,
 fol_name VARCHAR(255) NOT NULL,
 fol_description VARCHAR(1023) DEFAULT NULL,
 fol_grp_id INTEGER NOT NULL,
 fol_usr_id INTEGER NOT NULL,
 fol_access_group NUMERIC(1) DEFAULT 0 NOT NULL,
 fol_access_other NUMERIC(1) DEFAULT 0 NOT NULL,
 fol_date_creation TIMESTAMP NOT NULL DEFAULT LOCALTIMESTAMP,
 fol_date_modification TIMESTAMP NOT NULL DEFAULT LOCALTIMESTAMP,
 fol_last_modifier_usr_id INTEGER NOT NULL,
 fol_flags INTEGER DEFAULT 0 NOT NULL,
 fol_time_update TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
ALTER TABLE cec_folders ADD PRIMARY KEY (fol_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_unq_fol_id UNIQUE (fol_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_unq_fol_parent_id_fol_name UNIQUE (fol_parent_id, fol_name);
CREATE INDEX cec_idx_fol_obj_id ON cec_folders(fol_obj_id);
CREATE INDEX cec_idx_fol_grp_id ON cec_folders(fol_grp_id);
CREATE INDEX cec_idx_fol_usr_id ON cec_folders(fol_usr_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_fk_fol_parent_id     FOREIGN KEY (fol_parent_id) REFERENCES cec_folders(fol_id) ON DELETE CASCADE;
ALTER TABLE cec_folders ADD CONSTRAINT cec_fk_fol_obj_id 		FOREIGN KEY (fol_obj_id) 	REFERENCES cec_objects(obj_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_fk_fol_grp_id        FOREIGN KEY (fol_grp_id)    REFERENCES cec_groups(grp_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_fk_fol_usr_id        FOREIGN KEY (fol_usr_id)    REFERENCES cec_users(usr_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_fk_fol_last_modifier_usr_id FOREIGN KEY (fol_last_modifier_usr_id) REFERENCES cec_users(usr_id);
ALTER TABLE cec_folders ADD CONSTRAINT cec_chk_fol_parent_id    CHECK( (fol_name = 'ROOT' AND fol_parent_id IS NULL) OR (fol_name <> 'ROOT' AND fol_parent_id IS NOT NULL));
ALTER TABLE cec_folders ADD CONSTRAINT cec_chk_fol_access_group CHECK( fol_access_group BETWEEN 0 AND 3 );
ALTER TABLE cec_folders ADD CONSTRAINT cec_chk_fol_access_other CHECK( fol_access_other BETWEEN 0 AND 3 );

  
--
-- Definition des acces specifiques pour un dossier et un groupe donne
--
CREATE TABLE cec_accesses (
 acc_fol_id INTEGER NOT NULL,
 acc_grp_id INTEGER NOT NULL,
 acc_rights NUMERIC(1) DEFAULT 0 NOT NULL
);  

ALTER TABLE cec_accesses ADD PRIMARY KEY (acc_fol_id, acc_grp_id);
ALTER TABLE cec_accesses ADD CONSTRAINT cec_unq_acc_id UNIQUE (acc_fol_id, acc_grp_id);
CREATE INDEX cec_idx_acc_fol_id ON cec_accesses(acc_fol_id);
CREATE INDEX cec_idx_acc_grp_id ON cec_accesses(acc_grp_id);
ALTER TABLE cec_accesses ADD CONSTRAINT cec_fk_acc_fol_id FOREIGN KEY (acc_fol_id) REFERENCES cec_folders(fol_id) ON DELETE CASCADE;
ALTER TABLE cec_accesses ADD CONSTRAINT cec_fk_acc_grp_id FOREIGN KEY (acc_grp_id) REFERENCES cec_groups(grp_id);
ALTER TABLE cec_accesses ADD CONSTRAINT cec_chk_acc_rights CHECK( acc_rights BETWEEN 0 AND 3 );


--
-- Definition des relations entre dossiers
--
CREATE TABLE cec_relations (
 rel_src_id INTEGER NOT NULL,
 rel_dst_id INTEGER NOT NULL,
 rel_src_type VARCHAR(10) NOT NULL,
 rel_dst_type VARCHAR(10) NOT NULL
);
ALTER TABLE cec_relations ADD PRIMARY KEY (rel_src_id, rel_dst_id, rel_src_type, rel_dst_type);
ALTER TABLE cec_relations ADD CONSTRAINT cec_fk_rel_src_id FOREIGN KEY (rel_src_id) REFERENCES cec_folders(fol_id) ON DELETE CASCADE;
ALTER TABLE cec_relations ADD CONSTRAINT cec_fk_rel_dst_id FOREIGN KEY (rel_dst_id) REFERENCES cec_folders(fol_id) ;


--
-- Definition des evenements de la liste globale des evenements
--
CREATE TABLE cec_egli(
 egli_fol_id INTEGER NOT NULL,
 egli_obj_id INTEGER NOT NULL,
 egli_evt_name VARCHAR(255) NOT NULL,
 egli_evt_comment VARCHAR(255),
 egli_evt_type NUMERIC(4),
 egli_evt_data bytea NOT NULL
);
ALTER TABLE cec_egli ADD PRIMARY KEY (egli_obj_id, egli_evt_name);
ALTER TABLE cec_egli ADD CONSTRAINT cec_unq_egli UNIQUE (egli_obj_id, egli_evt_name);
CREATE INDEX cec_idx_egli_obj_id ON cec_egli(egli_obj_id);
ALTER TABLE cec_egli ADD CONSTRAINT cec_fk_egli_obj_id FOREIGN KEY (egli_obj_id) REFERENCES cec_objects(obj_id) ON DELETE CASCADE;


--
-- Ajout des sequences utilisees
--
CREATE SEQUENCE cec_licenses_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_groups_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_users_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_sessions_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 maxvalue 9999999999 CYCLE ;
CREATE SEQUENCE cec_preferencies_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_objects_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_obj_models_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_obj_properties_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;
CREATE SEQUENCE cec_folders_pk_seq INCREMENT BY 1 START WITH 1  minvalue 1 ;


COMMIT;
